# Authorization Quick Reference Guide

## Role Permissions Matrix

| Action | Super Admin | Company Admin | Rep |
|--------|-------------|---------------|-----|
| **Companies** | | | |
| View all companies | ✓ | ✗ | ✗ |
| View own company | ✓ | ✓ | ✓ |
| Create company | ✓ | ✗ | ✗ |
| Update company | ✓ | ✓ (own) | ✗ |
| Delete company | ✓ | ✗ | ✗ |
| **Pharmacies** | | | |
| View pharmacies | ✓ (all) | ✓ (company) | ✓ (company) |
| Create pharmacy | ✓ | ✓ | ✓ |
| Update pharmacy | ✓ | ✓ (company) | ✓ (company) |
| Delete pharmacy | ✓ | ✓ (company) | ✗ |
| **Products** | | | |
| View products | ✓ (all) | ✓ (company) | ✓ (company) |
| Create product | ✓ | ✓ | ✗ |
| Update product | ✓ | ✓ (company) | ✗ |
| Delete product | ✓ | ✓ (company) | ✗ |
| **Orders** | | | |
| View orders | ✓ (all) | ✓ (company) | ✓ (company) |
| Create order | ✓ | ✓ | ✓ |
| Update order | ✓ | ✓ (company) | ✓ (own) |
| Delete order | ✓ | ✓ (company) | ✗ |
| Resend email | ✓ | ✓ (company) | ✓ (own) |
| Regenerate PDF | ✓ | ✓ (company) | ✓ (own) |

## Controller Authorization Examples

### Basic CRUD Authorization

```php
// Index - View list
public function index(Request $request)
{
    $this->authorize('viewAny', Pharmacy::class);
    return PharmacyResource::collection(Pharmacy::all());
}

// Show - View single
public function show(Pharmacy $pharmacy)
{
    $this->authorize('view', $pharmacy);
    return new PharmacyResource($pharmacy);
}

// Store - Create
public function store(Request $request)
{
    $this->authorize('create', Pharmacy::class);

    $data = $request->validated();
    $data['company_id'] = auth()->user()->company_id; // CRITICAL

    return new PharmacyResource(Pharmacy::create($data));
}

// Update
public function update(Request $request, Pharmacy $pharmacy)
{
    $this->authorize('update', $pharmacy);
    $pharmacy->update($request->validated());
    return new PharmacyResource($pharmacy);
}

// Destroy
public function destroy(Pharmacy $pharmacy)
{
    $this->authorize('delete', $pharmacy);
    $pharmacy->delete();
    return response()->json(['message' => 'Deleted']);
}
```

### Custom Action Authorization

```php
public function resendEmail(Order $order)
{
    // Use custom policy method name
    $this->authorize('resendEmail', $order);

    SendOrderEmailJob::dispatch($order);
    return response()->json(['message' => 'Email queued']);
}
```

## Gate Usage Examples

```php
use Illuminate\Support\Facades\Gate;

// Check if super admin
if (Gate::allows('isSuperAdmin')) {
    // Admin-only code
}

// Check if company admin or higher
if (Gate::allows('isCompanyAdmin')) {
    // Admin functionality
}

// Check if model belongs to user's company
if (Gate::allows('belongsToCompany', $pharmacy)) {
    // Access allowed
}

// Deny access if not authorized
Gate::authorize('manageCatalog');
```

## Middleware Usage

### Apply to Route Groups

```php
// Full protection
Route::middleware(['auth:sanctum', 'ensure.active.user', 'ensure.company.access'])
    ->group(function () {
        Route::apiResource('pharmacies', PharmacyController::class);
    });

// Active user check only
Route::middleware(['auth:sanctum', 'ensure.active.user'])
    ->group(function () {
        Route::get('/dashboard', [DashboardController::class, 'index']);
    });
```

### Apply to Individual Routes

```php
Route::get('/pharmacies/{pharmacy}', [PharmacyController::class, 'show'])
    ->middleware(['auth:sanctum', 'ensure.company.access']);
```

## Common Patterns

### Auto-assigning company_id on Create

```php
public function store(Request $request)
{
    $this->authorize('create', Product::class);

    $product = Product::create([
        'company_id' => auth()->user()->company_id, // From authenticated user
        'name' => $request->name,
        'price' => $request->price,
        // ... other fields
    ]);

    return new ProductResource($product);
}
```

### Auto-assigning rep_id for Orders

```php
public function store(Request $request)
{
    $this->authorize('create', Order::class);

    $order = Order::create([
        'company_id' => auth()->user()->company_id,
        'rep_id' => auth()->id(), // Current user as rep
        'pharmacy_id' => $request->pharmacy_id,
        // ... other fields
    ]);

    return new OrderResource($order);
}
```

### Checking Ownership for Updates

```php
public function update(Request $request, Order $order)
{
    // Policy automatically checks:
    // - Company admin can update any order in their company
    // - Rep can only update orders they created (rep_id matches)
    $this->authorize('update', $order);

    $order->update($request->validated());
    return new OrderResource($order);
}
```

## Testing Authorization

### Feature Test Examples

```php
public function test_rep_cannot_delete_pharmacy()
{
    $company = Company::factory()->create();
    $rep = User::factory()->rep()->for($company)->create();
    $pharmacy = Pharmacy::factory()->for($company)->create();

    $response = $this->actingAs($rep)
        ->deleteJson("/api/v1/pharmacies/{$pharmacy->id}");

    $response->assertStatus(403); // Forbidden
}

public function test_user_cannot_access_other_company_data()
{
    $company1 = Company::factory()->create();
    $company2 = Company::factory()->create();

    $user = User::factory()->for($company1)->create();
    $pharmacy = Pharmacy::factory()->for($company2)->create();

    $response = $this->actingAs($user)
        ->getJson("/api/v1/pharmacies/{$pharmacy->id}");

    $response->assertStatus(404); // Global scope filters it out
}

public function test_company_admin_can_manage_company_products()
{
    $company = Company::factory()->create();
    $admin = User::factory()->companyAdmin()->for($company)->create();
    $product = Product::factory()->for($company)->create();

    $response = $this->actingAs($admin)
        ->putJson("/api/v1/products/{$product->id}", [
            'name' => 'Updated Name',
        ]);

    $response->assertStatus(200);
}
```

## Policy Methods Reference

### Standard Methods
- `viewAny(User $user)` - Can view list/index
- `view(User $user, Model $model)` - Can view specific instance
- `create(User $user)` - Can create new instances
- `update(User $user, Model $model)` - Can update instance
- `delete(User $user, Model $model)` - Can delete instance
- `restore(User $user, Model $model)` - Can restore soft-deleted instance
- `forceDelete(User $user, Model $model)` - Can permanently delete instance

### Custom Methods (OrderPolicy)
- `resendEmail(User $user, Order $order)` - Can resend order email
- `regeneratePDF(User $user, Order $order)` - Can regenerate order PDF

### Custom Methods (ProductPolicy)
- `toggleActive(User $user, Product $product)` - Can activate/deactivate product

## Important Security Notes

### ⚠️ Always Auto-assign company_id

```php
// CORRECT ✓
$data = $request->validated();
$data['company_id'] = auth()->user()->company_id;
$model = Model::create($data);

// WRONG ✗ - User could inject different company_id
$model = Model::create($request->all());
```

### ⚠️ Trust Global Scopes but Verify

Global scopes automatically filter by company_id, but:
- Always add authorization checks in controllers
- Don't bypass global scopes unless absolutely necessary
- Use `withoutGlobalScope('company')` sparingly

### ⚠️ 403 vs 404 Responses

- **403 Forbidden:** Authorization failed (user authenticated but lacks permission)
- **404 Not Found:** Resource not found (usually due to global scope filtering)

Both prevent data leakage but 404 doesn't reveal that resource exists.

## Debugging Authorization Issues

### Check User Role
```php
dd(auth()->user()->role); // super_admin, company_admin, or rep
```

### Check User Company
```php
dd(auth()->user()->company_id);
```

### Check Model Company
```php
dd($pharmacy->company_id);
```

### Test Policy Directly
```php
dd(Gate::allows('view', $pharmacy));
dd(auth()->user()->can('update', $product));
```

### Disable Global Scope Temporarily (for debugging only)
```php
$allPharmacies = Pharmacy::withoutGlobalScope('company')->get();
dd($allPharmacies); // See all pharmacies across companies
```

## Files Reference

- **Policies:** `/home/orderlyxdev/public_html/HaleonAPI/app/Policies/`
- **Middleware:** `/home/orderlyxdev/public_html/HaleonAPI/app/Http/Middleware/`
- **Service Provider:** `/home/orderlyxdev/public_html/HaleonAPI/app/Providers/AuthServiceProvider.php`
- **Documentation:** `/home/orderlyxdev/public_html/HaleonAPI/AUTHORIZATION.md`
