<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Company;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class CompanyController extends Controller
{
    /**
     * Display a listing of companies (super_admin only).
     */
    public function index(Request $request): Response
    {
        // Only super_admin can view all companies
        $this->authorize('viewAny', Company::class);

        $companies = Company::query()
            ->withCount(['users', 'products', 'orders'])
            ->when($request->input('search'), function ($query, $search) {
                $query->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            })
            ->when($request->input('status'), function ($query, $status) {
                $query->where('status', $status);
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        return Inertia::render('Admin/Companies/Index', [
            'companies' => $companies,
            'filters' => $request->only(['search', 'status']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Companies', 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for creating a new company.
     */
    public function create(): Response
    {
        $this->authorize('create', Company::class);

        return Inertia::render('Admin/Companies/Create', [
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Companies', 'url' => route('admin.companies.index')],
                ['label' => 'Create', 'url' => null],
            ],
        ]);
    }

    /**
     * Store a newly created company in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Company::class);

        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:companies',
            'email' => 'required|email|max:255|unique:companies',
            'phone' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
            'status' => 'required|in:active,inactive',
            'max_users' => 'nullable|integer|min:1',
            'max_products' => 'nullable|integer|min:1',
        ]);

        $company = Company::create($validated);

        return redirect()
            ->route('admin.companies.show', $company)
            ->with('success', 'Company created successfully.');
    }

    /**
     * Display the specified company with statistics.
     */
    public function show(Request $request, Company $company): Response
    {
        $this->authorize('view', $company);

        $company->loadCount([
            'users',
            'products',
            'orders',
            'pharmacies',
        ]);

        // Get statistics
        $stats = [
            'total_users' => $company->users_count,
            'active_users' => $company->users()->where('status', 'active')->count(),
            'total_products' => $company->products_count,
            'active_products' => $company->products()->where('active', true)->count(),
            'total_orders' => $company->orders_count,
            'completed_orders' => $company->orders()->where('status', 'completed')->count(),
            'total_pharmacies' => $company->pharmacies_count,
            'active_pharmacies' => $company->pharmacies()->where('status', 'active')->count(),
        ];

        // Recent activity
        $recentOrders = $company->orders()
            ->latest()
            ->limit(10)
            ->with(['pharmacy', 'user'])
            ->get();

        $recentUsers = $company->users()
            ->latest()
            ->limit(5)
            ->get();

        return Inertia::render('Admin/Companies/Show', [
            'company' => $company,
            'stats' => $stats,
            'recentOrders' => $recentOrders,
            'recentUsers' => $recentUsers,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Companies', 'url' => route('admin.companies.index')],
                ['label' => $company->name, 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for editing the specified company.
     */
    public function edit(Company $company): Response
    {
        $this->authorize('update', $company);

        return Inertia::render('Admin/Companies/Edit', [
            'company' => $company,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Companies', 'url' => route('admin.companies.index')],
                ['label' => $company->name, 'url' => route('admin.companies.show', $company)],
                ['label' => 'Edit', 'url' => null],
            ],
        ]);
    }

    /**
     * Update the specified company in storage.
     */
    public function update(Request $request, Company $company): RedirectResponse
    {
        $this->authorize('update', $company);

        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:companies,name,' . $company->id,
            'email' => 'required|email|max:255|unique:companies,email,' . $company->id,
            'phone' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
            'status' => 'required|in:active,inactive',
            'max_users' => 'nullable|integer|min:1',
            'max_products' => 'nullable|integer|min:1',
        ]);

        $company->update($validated);

        return redirect()
            ->route('admin.companies.show', $company)
            ->with('success', 'Company updated successfully.');
    }

    /**
     * Soft delete the specified company from storage.
     */
    public function destroy(Company $company): RedirectResponse
    {
        $this->authorize('delete', $company);

        // Soft delete the company
        $company->delete();

        return redirect()
            ->route('admin.companies.index')
            ->with('success', 'Company deleted successfully.');
    }
}
