<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class OrderController extends Controller
{
    /**
     * Display a listing of orders for the current company with filters.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Base query - scope by company unless super_admin
        $query = $user->role === 'super_admin'
            ? Order::query()
            : Order::where('company_id', $companyId);

        $orders = $query
            ->with(['pharmacy', 'rep', 'items'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('order_number', 'like', "%{$search}%")
                        ->orWhereHas('pharmacy', function ($pharmacyQuery) use ($search) {
                            $pharmacyQuery->where('pharmacy_name', 'like', "%{$search}%");
                        });
                });
            })
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->input('pharmacy_id'), function ($q, $pharmacyId) {
                $q->where('pharmacy_id', $pharmacyId);
            })
            ->when($request->input('user_id'), function ($q, $userId) {
                $q->where('user_id', $userId);
            })
            ->when($request->input('date_from'), function ($q, $dateFrom) {
                $q->whereDate('created_at', '>=', $dateFrom);
            })
            ->when($request->input('date_to'), function ($q, $dateTo) {
                $q->whereDate('created_at', '<=', $dateTo);
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Get filter options
        $pharmacies = $user->role === 'super_admin'
            ? \App\Models\Pharmacy::orderBy('pharmacy_name')->get()
            : \App\Models\Pharmacy::where('company_id', $companyId)->orderBy('pharmacy_name')->get();

        $reps = $user->role === 'super_admin'
            ? \App\Models\User::where('role', 'rep')->orderBy('name')->get()
            : \App\Models\User::where('role', 'rep')->where('company_id', $companyId)->orderBy('name')->get();

        return Inertia::render('Admin/Orders/Index', [
            'orders' => $orders,
            'pharmacies' => $pharmacies,
            'reps' => $reps,
            'filters' => $request->only(['search', 'status', 'pharmacy_id', 'user_id', 'date_from', 'date_to']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Orders', 'url' => null],
            ],
        ]);
    }

    /**
     * Display the specified order with all details and items.
     */
    public function show(Request $request, Order $order): Response
    {
        $user = $request->user();

        // Ensure order belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $order->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $order->load([
            'pharmacy',
            'user',
            'items.product.brand',
            'pdf',
        ]);

        // Group items by brand for display
        $itemsByBrand = $order->items->groupBy(function ($item) {
            return $item->product->brand->name ?? 'Unknown Brand';
        });

        return Inertia::render('Admin/Orders/Show', [
            'order' => $order,
            'itemsByBrand' => $itemsByBrand,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Orders', 'url' => route('admin.orders.index')],
                ['label' => 'Order #' . $order->id, 'url' => null],
            ],
        ]);
    }

    /**
     * Update the status of the specified order.
     */
    public function updateStatus(Request $request, Order $order): RedirectResponse
    {
        $user = $request->user();

        // Ensure order belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $order->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'status' => 'required|in:pending,processing,completed,failed',
            'notes' => 'nullable|string|max:1000',
        ]);

        $order->update($validated);

        // If marking as completed, set processed_at
        if ($validated['status'] === 'completed') {
            $order->markAsCompleted();
        }

        return redirect()
            ->route('admin.orders.show', $order)
            ->with('success', 'Order status updated successfully.');
    }

    /**
     * Export orders to CSV.
     */
    public function export(Request $request): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Base query - scope by company unless super_admin
        $query = $user->role === 'super_admin'
            ? Order::query()
            : Order::where('company_id', $companyId);

        // Apply same filters as index
        $orders = $query
            ->with(['pharmacy', 'user'])
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->input('pharmacy_id'), function ($q, $pharmacyId) {
                $q->where('pharmacy_id', $pharmacyId);
            })
            ->when($request->input('user_id'), function ($q, $userId) {
                $q->where('user_id', $userId);
            })
            ->when($request->input('date_from'), function ($q, $dateFrom) {
                $q->whereDate('created_at', '>=', $dateFrom);
            })
            ->when($request->input('date_to'), function ($q, $dateTo) {
                $q->whereDate('created_at', '<=', $dateTo);
            })
            ->latest()
            ->get();

        $filename = 'orders-' . now()->format('Y-m-d-His') . '.csv';

        return response()->streamDownload(function () use ($orders) {
            $handle = fopen('php://output', 'w');

            // Add CSV headers
            fputcsv($handle, [
                'Order ID',
                'Pharmacy',
                'Rep',
                'Total Items',
                'Status',
                'Created At',
                'Processed At',
                'Notes',
            ]);

            foreach ($orders as $order) {
                fputcsv($handle, [
                    $order->id,
                    $order->pharmacy->name ?? 'N/A',
                    $order->user->name ?? 'N/A',
                    $order->total_items,
                    $order->status,
                    $order->created_at->format('Y-m-d H:i:s'),
                    $order->processed_at?->format('Y-m-d H:i:s') ?? 'N/A',
                    $order->notes ?? '',
                ]);
            }

            fclose($handle);
        }, $filename, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * Export orders to PDF.
     */
    public function exportPdf(Request $request): \Illuminate\Http\Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Base query - scope by company unless super_admin
        $query = $user->role === 'super_admin'
            ? Order::query()
            : Order::where('company_id', $companyId);

        // Apply same filters as index
        $orders = $query
            ->with(['pharmacy', 'user'])
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->input('pharmacy_id'), function ($q, $pharmacyId) {
                $q->where('pharmacy_id', $pharmacyId);
            })
            ->when($request->input('user_id'), function ($q, $userId) {
                $q->where('user_id', $userId);
            })
            ->when($request->input('date_from'), function ($q, $dateFrom) {
                $q->whereDate('created_at', '>=', $dateFrom);
            })
            ->when($request->input('date_to'), function ($q, $dateTo) {
                $q->whereDate('created_at', '<=', $dateTo);
            })
            ->latest()
            ->get();

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.orders-report', [
            'orders' => $orders,
            'companyName' => $user->company->name ?? 'Company',
            'generatedAt' => now()->format('Y-m-d H:i:s'),
            'filters' => $request->only(['status', 'pharmacy_id', 'user_id', 'date_from', 'date_to']),
        ]);

        $filename = 'orders-report-' . now()->format('Y-m-d-His') . '.pdf';

        return $pdf->download($filename);
    }
}
