<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Pharmacy;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Inertia\Response;

class PharmacyController extends Controller
{
    /**
     * Display a listing of pharmacies for the current company.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Base query - scope by company unless super_admin
        $query = $user->role === 'super_admin'
            ? Pharmacy::query()
            : Pharmacy::where('company_id', $companyId);

        $pharmacies = $query
            ->with(['company', 'wholesalers'])
            ->withCount(['orders'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('pharmacy_name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%")
                        ->orWhere('registration_number', 'like', "%{$search}%");
                });
            })
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->input('wholesaler_id'), function ($q, $wholesalerId) {
                $q->whereHas('wholesalers', function ($query) use ($wholesalerId) {
                    $query->where('wholesalers.id', $wholesalerId);
                });
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        return Inertia::render('Admin/Pharmacies/Index', [
            'pharmacies' => $pharmacies,
            'filters' => $request->only(['search', 'status', 'wholesaler_id']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Pharmacies', 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for creating a new pharmacy.
     */
    public function create(): Response
    {
        return Inertia::render('Admin/Pharmacies/Create', [
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Pharmacies', 'url' => route('admin.pharmacies.index')],
                ['label' => 'Create', 'url' => null],
            ],
        ]);
    }

    /**
     * Store a newly created pharmacy in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $user = $request->user();

        $validated = $request->validate([
            'pharmacy_name' => 'required|string|max:255',
            'registration_number' => 'nullable|string|max:100|unique:pharmacies',
            'contact_person' => 'nullable|string|max:255',
            'phone' => 'required|string|max:50',
            'email' => 'required|email|max:255|unique:pharmacies',
            'physical_address' => 'nullable|string|max:500',
            'delivery_address' => 'nullable|string|max:500',
            'status' => 'required|in:active,inactive',
        ]);

        // Auto-assign company_id from authenticated user
        $validated['company_id'] = $user->role === 'super_admin'
            ? ($request->input('company_id') ?? $user->company_id)
            : $user->company_id;

        $validated['created_by_rep_id'] = $user->id;

        $pharmacy = Pharmacy::create($validated);

        return redirect()
            ->route('admin.pharmacies.show', $pharmacy)
            ->with('success', 'Pharmacy created successfully.');
    }

    /**
     * Display the specified pharmacy with order history.
     */
    public function show(Request $request, Pharmacy $pharmacy): Response
    {
        $user = $request->user();

        // Ensure pharmacy belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $pharmacy->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $pharmacy->load(['company', 'wholesalers']);
        $pharmacy->loadCount(['orders']);

        // Order statistics
        $stats = [
            'total_orders' => $pharmacy->orders_count,
            'completed_orders' => $pharmacy->orders()->where('status', 'completed')->count(),
            'pending_orders' => $pharmacy->orders()->where('status', 'pending')->count(),
            'failed_orders' => $pharmacy->orders()->where('status', 'failed')->count(),
            'total_items_ordered' => $pharmacy->orders()->sum('total_items'),
            'orders_this_month' => $pharmacy->orders()
                ->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->count(),
            'orders_this_week' => $pharmacy->orders()
                ->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])
                ->count(),
            'last_order_date' => $pharmacy->orders()
                ->latest()
                ->first()?->created_at,
        ];

        // Order history
        $orders = $pharmacy->orders()
            ->with(['user', 'items'])
            ->latest()
            ->paginate(10);

        return Inertia::render('Admin/Pharmacies/Show', [
            'pharmacy' => $pharmacy,
            'stats' => $stats,
            'orders' => $orders,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Pharmacies', 'url' => route('admin.pharmacies.index')],
                ['label' => $pharmacy->pharmacy_name, 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for editing the specified pharmacy.
     */
    public function edit(Request $request, Pharmacy $pharmacy): Response
    {
        $user = $request->user();

        // Ensure pharmacy belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $pharmacy->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Get wholesalers for dropdown
        $wholesalers = $user->role === 'super_admin'
            ? \App\Models\Wholesaler::orderBy('wholesaler_name')->get()
            : \App\Models\Wholesaler::where('company_id', $user->company_id)->orderBy('wholesaler_name')->get();

        return Inertia::render('Admin/Pharmacies/Edit', [
            'pharmacy' => $pharmacy,
            'wholesalers' => $wholesalers,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Pharmacies', 'url' => route('admin.pharmacies.index')],
                ['label' => $pharmacy->pharmacy_name, 'url' => route('admin.pharmacies.show', $pharmacy)],
                ['label' => 'Edit', 'url' => null],
            ],
        ]);
    }

    /**
     * Update the specified pharmacy in storage.
     */
    public function update(Request $request, Pharmacy $pharmacy): RedirectResponse
    {
        $user = $request->user();

        // Ensure pharmacy belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $pharmacy->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('pharmacies')->ignore($pharmacy->id),
            ],
            'phone' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
            'license_number' => [
                'nullable',
                'string',
                'max:100',
                Rule::unique('pharmacies')->ignore($pharmacy->id),
            ],
            'wholesaler_id' => 'nullable|exists:wholesalers,id',
            'status' => 'required|in:active,inactive',
        ]);

        $pharmacy->update($validated);

        return redirect()
            ->route('admin.pharmacies.show', $pharmacy)
            ->with('success', 'Pharmacy updated successfully.');
    }

    /**
     * Soft delete the specified pharmacy from storage.
     */
    public function destroy(Request $request, Pharmacy $pharmacy): RedirectResponse
    {
        $user = $request->user();

        // Ensure pharmacy belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $pharmacy->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $pharmacy->delete();

        return redirect()
            ->route('admin.pharmacies.index')
            ->with('success', 'Pharmacy deleted successfully.');
    }
}
