<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Brand;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Inertia\Response;

class ProductController extends Controller
{
    /**
     * Display a listing of products for the current company.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Base query - scope by company unless super_admin
        $query = $user->role === 'super_admin'
            ? Product::query()
            : Product::where('company_id', $companyId);

        $products = $query
            ->with(['brand', 'company'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('name', 'like', "%{$search}%")
                        ->orWhere('barcode', 'like', "%{$search}%")
                        ->orWhere('size', 'like', "%{$search}%");
                });
            })
            ->when($request->input('brand_id'), function ($q, $brandId) {
                $q->where('brand_id', $brandId);
            })
            ->when($request->input('active') !== null, function ($q) use ($request) {
                $q->where('active', $request->input('active'));
            })
            ->orderBy('sort_order')
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Get brands for filter dropdown
        $brands = $user->role === 'super_admin'
            ? Brand::orderBy('name')->get()
            : Brand::where('company_id', $companyId)->orderBy('name')->get();

        return Inertia::render('Admin/Products/Index', [
            'products' => $products,
            'brands' => $brands,
            'filters' => $request->only(['search', 'brand_id', 'active']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Products', 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for creating a new product.
     */
    public function create(Request $request): Response
    {
        $user = $request->user();

        // Get brands for the current company
        $brands = $user->role === 'super_admin'
            ? Brand::orderBy('name')->get()
            : Brand::where('company_id', $user->company_id)->orderBy('name')->get();

        return Inertia::render('Admin/Products/Create', [
            'brands' => $brands,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Products', 'url' => route('admin.products.index')],
                ['label' => 'Create', 'url' => null],
            ],
        ]);
    }

    /**
     * Store a newly created product in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $user = $request->user();

        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'size' => 'nullable|string|max:100',
            'price' => 'nullable|numeric|min:0|max:99999999.99',
            'barcode' => 'required|string|max:100|unique:products',
            'active' => 'required|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        // Auto-assign company_id from authenticated user
        $validated['company_id'] = $user->role === 'super_admin'
            ? ($request->input('company_id') ?? $user->company_id)
            : $user->company_id;

        $product = Product::create($validated);

        return redirect()
            ->route('admin.products.show', $product)
            ->with('success', 'Product created successfully.');
    }

    /**
     * Display the specified product.
     */
    public function show(Request $request, Product $product): Response
    {
        $user = $request->user();

        // Ensure product belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $product->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $product->load(['brand', 'company']);
        $product->loadCount(['orderItems']);

        // Get order statistics
        $stats = [
            'total_orders' => $product->orderItems_count,
            'total_quantity_ordered' => $product->orderItems()->sum('quantity'),
            'last_ordered_at' => $product->orderItems()
                ->latest()
                ->first()?->created_at,
        ];

        return Inertia::render('Admin/Products/Show', [
            'product' => $product,
            'stats' => $stats,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Products', 'url' => route('admin.products.index')],
                ['label' => $product->name, 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for editing the specified product.
     */
    public function edit(Request $request, Product $product): Response
    {
        $user = $request->user();

        // Ensure product belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $product->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Get brands for the current company
        $brands = $user->role === 'super_admin'
            ? Brand::orderBy('name')->get()
            : Brand::where('company_id', $user->company_id)->orderBy('name')->get();

        return Inertia::render('Admin/Products/Edit', [
            'product' => $product,
            'brands' => $brands,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Products', 'url' => route('admin.products.index')],
                ['label' => $product->name, 'url' => route('admin.products.show', $product)],
                ['label' => 'Edit', 'url' => null],
            ],
        ]);
    }

    /**
     * Update the specified product in storage.
     */
    public function update(Request $request, Product $product): RedirectResponse
    {
        $user = $request->user();

        // Ensure product belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $product->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'size' => 'nullable|string|max:100',
            'price' => 'nullable|numeric|min:0|max:99999999.99',
            'barcode' => [
                'required',
                'string',
                'max:100',
                Rule::unique('products')->ignore($product->id),
            ],
            'active' => 'required|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        $product->update($validated);

        return redirect()
            ->route('admin.products.show', $product)
            ->with('success', 'Product updated successfully.');
    }

    /**
     * Soft delete the specified product from storage.
     */
    public function destroy(Request $request, Product $product): RedirectResponse
    {
        $user = $request->user();

        // Ensure product belongs to current company (unless super_admin)
        if ($user->role !== 'super_admin' && $product->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $product->delete();

        return redirect()
            ->route('admin.products.index')
            ->with('success', 'Product deleted successfully.');
    }

    /**
     * Show the CSV import form.
     */
    public function import(): Response
    {
        return Inertia::render('Admin/Products/Import', [
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Products', 'url' => route('admin.products.index')],
                ['label' => 'Import', 'url' => null],
            ],
        ]);
    }

    /**
     * Process CSV file import.
     */
    public function processImport(Request $request): RedirectResponse
    {
        $user = $request->user();

        $request->validate([
            'csv_file' => 'required|file|mimes:csv,txt|max:10240', // 10MB max
        ]);

        $file = $request->file('csv_file');
        $path = $file->store('imports', 'local');

        try {
            $fullPath = Storage::path($path);
            $handle = fopen($fullPath, 'r');

            $imported = 0;
            $updated = 0;
            $skipped = 0;
            $currentBrand = null;

            // Skip header rows (first 16 rows as per import command)
            for ($i = 0; $i < 16; $i++) {
                fgetcsv($handle);
            }

            while (($row = fgetcsv($handle)) !== false) {
                // Check if this is a brand header (no barcode/size)
                if (empty($row[2]) && empty($row[3])) {
                    // This is a brand header
                    $brandName = trim($row[1]);
                    if ($brandName) {
                        $currentBrand = Brand::firstOrCreate(
                            [
                                'name' => $brandName,
                                'company_id' => $user->company_id,
                            ],
                            [
                                'active' => true,
                                'sort_order' => 0,
                            ]
                        );
                    }
                    continue;
                }

                // This is a product row
                if ($currentBrand && !empty($row[2])) {
                    $barcode = trim($row[2]);
                    $productName = trim($row[1]);
                    $size = trim($row[3] ?? '');

                    if ($barcode && $productName) {
                        $product = Product::updateOrCreate(
                            [
                                'barcode' => $barcode,
                                'company_id' => $user->company_id,
                            ],
                            [
                                'brand_id' => $currentBrand->id,
                                'name' => $productName,
                                'size' => $size ?: null,
                                'active' => true,
                                'sort_order' => 0,
                            ]
                        );

                        if ($product->wasRecentlyCreated) {
                            $imported++;
                        } else {
                            $updated++;
                        }
                    } else {
                        $skipped++;
                    }
                } else {
                    $skipped++;
                }
            }

            fclose($handle);
            Storage::delete($path);

            return redirect()
                ->route('admin.products.index')
                ->with('success', "Import completed: {$imported} created, {$updated} updated, {$skipped} skipped.");
        } catch (\Exception $e) {
            if (isset($path)) {
                Storage::delete($path);
            }

            return redirect()
                ->back()
                ->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Export products to CSV.
     */
    public function export(Request $request): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Base query - scope by company unless super_admin
        $query = $user->role === 'super_admin'
            ? Product::query()
            : Product::where('company_id', $companyId);

        $products = $query
            ->with(['brand'])
            ->orderBy('brand_id')
            ->orderBy('sort_order')
            ->get();

        $filename = 'products-' . now()->format('Y-m-d-His') . '.csv';

        return response()->streamDownload(function () use ($products) {
            $handle = fopen('php://output', 'w');

            // Add CSV headers
            fputcsv($handle, ['ID', 'Brand', 'Product Name', 'Size', 'Price', 'Barcode', 'Active', 'Sort Order']);

            foreach ($products as $product) {
                fputcsv($handle, [
                    $product->id,
                    $product->brand->name ?? '',
                    $product->name,
                    $product->size,
                    $product->price ? number_format($product->price, 2) : '',
                    $product->barcode,
                    $product->active ? 'Yes' : 'No',
                    $product->sort_order,
                ]);
            }

            fclose($handle);
        }, $filename, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }
}
