<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use App\Models\User;
use App\Models\Pharmacy;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class ReportController extends Controller
{
    /**
     * Display the reports landing page.
     */
    public function index(): Response
    {
        return Inertia::render('Admin/Reports/Index', [
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Reports', 'url' => null],
            ],
        ]);
    }

    /**
     * Generate sales performance report by rep.
     */
    public function salesByRep(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Date range filter
        $dateFrom = $request->input('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->input('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Base query for reps
        $repsQuery = $user->role === 'super_admin'
            ? User::where('role', 'rep')
            : User::where('role', 'rep')->where('company_id', $companyId);

        $repPerformance = $repsQuery
            ->withCount([
                'orders' => function ($query) use ($dateFrom, $dateTo) {
                    $query->whereBetween('created_at', [$dateFrom, $dateTo]);
                },
                'orders as completed_orders_count' => function ($query) use ($dateFrom, $dateTo) {
                    $query->where('status', 'completed')
                        ->whereBetween('created_at', [$dateFrom, $dateTo]);
                },
            ])
            ->with(['orders' => function ($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('created_at', [$dateFrom, $dateTo]);
            }])
            ->get()
            ->map(function ($rep) use ($dateFrom, $dateTo) {
                $totalItems = $rep->orders()
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->sum('total_items');

                $completedItems = $rep->orders()
                    ->where('status', 'completed')
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->sum('total_items');

                return [
                    'id' => $rep->id,
                    'name' => $rep->name,
                    'email' => $rep->email,
                    'total_orders' => $rep->orders_count,
                    'completed_orders' => $rep->completed_orders_count,
                    'total_items' => $totalItems,
                    'completed_items' => $completedItems,
                    'success_rate' => $rep->orders_count > 0
                        ? round(($rep->completed_orders_count / $rep->orders_count) * 100, 2)
                        : 0,
                ];
            })
            ->sortByDesc('total_orders')
            ->values();

        return Inertia::render('Admin/Reports/SalesByRep', [
            'repPerformance' => $repPerformance,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Reports', 'url' => route('admin.reports.index')],
                ['label' => 'Sales by Rep', 'url' => null],
            ],
        ]);
    }

    /**
     * Generate order patterns report by pharmacy.
     */
    public function salesByPharmacy(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Date range filter
        $dateFrom = $request->input('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->input('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Base query for pharmacies
        $pharmaciesQuery = $user->role === 'super_admin'
            ? Pharmacy::query()
            : Pharmacy::where('company_id', $companyId);

        $pharmacyPerformance = $pharmaciesQuery
            ->withCount([
                'orders' => function ($query) use ($dateFrom, $dateTo) {
                    $query->whereBetween('created_at', [$dateFrom, $dateTo]);
                },
                'orders as completed_orders_count' => function ($query) use ($dateFrom, $dateTo) {
                    $query->where('status', 'completed')
                        ->whereBetween('created_at', [$dateFrom, $dateTo]);
                },
            ])
            ->with(['orders' => function ($query) use ($dateFrom, $dateTo) {
                $query->whereBetween('created_at', [$dateFrom, $dateTo]);
            }])
            ->get()
            ->map(function ($pharmacy) use ($dateFrom, $dateTo) {
                $totalItems = $pharmacy->orders()
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->sum('total_items');

                $lastOrderDate = $pharmacy->orders()
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->latest()
                    ->first()?->created_at;

                return [
                    'id' => $pharmacy->id,
                    'name' => $pharmacy->name,
                    'email' => $pharmacy->email,
                    'total_orders' => $pharmacy->orders_count,
                    'completed_orders' => $pharmacy->completed_orders_count,
                    'total_items' => $totalItems,
                    'last_order_date' => $lastOrderDate?->format('Y-m-d H:i:s'),
                    'average_order_size' => $pharmacy->orders_count > 0
                        ? round($totalItems / $pharmacy->orders_count, 2)
                        : 0,
                ];
            })
            ->sortByDesc('total_orders')
            ->values();

        return Inertia::render('Admin/Reports/SalesByPharmacy', [
            'pharmacyPerformance' => $pharmacyPerformance,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Reports', 'url' => route('admin.reports.index')],
                ['label' => 'Sales by Pharmacy', 'url' => null],
            ],
        ]);
    }

    /**
     * Generate product performance report (top selling products).
     */
    public function productPerformance(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Date range filter
        $dateFrom = $request->input('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->input('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Get product performance data
        $productPerformance = OrderItem::query()
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('products', 'order_items.product_id', '=', 'products.id')
            ->join('brands', 'products.brand_id', '=', 'brands.id')
            ->when($user->role !== 'super_admin', function ($query) use ($companyId) {
                $query->where('orders.company_id', $companyId);
            })
            ->whereBetween('orders.created_at', [$dateFrom, $dateTo])
            ->select([
                'products.id',
                'products.name',
                'products.size',
                'products.barcode',
                'brands.name as brand_name',
                DB::raw('SUM(order_items.quantity) as total_quantity'),
                DB::raw('COUNT(DISTINCT order_items.order_id) as order_count'),
                DB::raw('COUNT(DISTINCT orders.pharmacy_id) as unique_pharmacies'),
            ])
            ->groupBy([
                'products.id',
                'products.name',
                'products.size',
                'products.barcode',
                'brands.name',
            ])
            ->orderByDesc('total_quantity')
            ->limit(50)
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'name' => $item->name,
                    'size' => $item->size,
                    'barcode' => $item->barcode,
                    'brand_name' => $item->brand_name,
                    'total_quantity' => $item->total_quantity,
                    'order_count' => $item->order_count,
                    'unique_pharmacies' => $item->unique_pharmacies,
                    'average_per_order' => $item->order_count > 0
                        ? round($item->total_quantity / $item->order_count, 2)
                        : 0,
                ];
            });

        return Inertia::render('Admin/Reports/ProductPerformance', [
            'productPerformance' => $productPerformance,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Reports', 'url' => route('admin.reports.index')],
                ['label' => 'Product Performance', 'url' => null],
            ],
        ]);
    }

    /**
     * Export report to PDF or Excel.
     */
    public function export(Request $request): \Illuminate\Http\Response|\Symfony\Component\HttpFoundation\StreamedResponse
    {
        $validated = $request->validate([
            'report_type' => 'required|in:sales_by_rep,sales_by_pharmacy,product_performance',
            'format' => 'required|in:pdf,csv',
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date',
        ]);

        $reportType = $validated['report_type'];
        $format = $validated['format'];

        // Get report data based on type
        $data = $this->getReportData($request, $reportType);

        if ($format === 'csv') {
            return $this->exportCsv($data, $reportType);
        } else {
            return $this->exportPdf($data, $reportType, $request);
        }
    }

    /**
     * Get report data based on report type.
     */
    private function getReportData(Request $request, string $reportType): array
    {
        $user = $request->user();
        $dateFrom = $request->input('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->input('date_to', now()->endOfMonth()->format('Y-m-d'));

        switch ($reportType) {
            case 'sales_by_rep':
                // Re-use the logic from salesByRep method
                $repsQuery = $user->role === 'super_admin'
                    ? User::where('role', 'rep')
                    : User::where('role', 'rep')->where('company_id', $user->company_id);

                return $repsQuery
                    ->withCount([
                        'orders' => function ($query) use ($dateFrom, $dateTo) {
                            $query->whereBetween('created_at', [$dateFrom, $dateTo]);
                        },
                    ])
                    ->get()
                    ->toArray();

            case 'sales_by_pharmacy':
                $pharmaciesQuery = $user->role === 'super_admin'
                    ? Pharmacy::query()
                    : Pharmacy::where('company_id', $user->company_id);

                return $pharmaciesQuery
                    ->withCount([
                        'orders' => function ($query) use ($dateFrom, $dateTo) {
                            $query->whereBetween('created_at', [$dateFrom, $dateTo]);
                        },
                    ])
                    ->get()
                    ->toArray();

            case 'product_performance':
                return OrderItem::query()
                    ->join('orders', 'order_items.order_id', '=', 'orders.id')
                    ->join('products', 'order_items.product_id', '=', 'products.id')
                    ->when($user->role !== 'super_admin', function ($query) use ($user) {
                        $query->where('orders.company_id', $user->company_id);
                    })
                    ->whereBetween('orders.created_at', [$dateFrom, $dateTo])
                    ->select([
                        'products.name',
                        DB::raw('SUM(order_items.quantity) as total_quantity'),
                    ])
                    ->groupBy('products.name')
                    ->get()
                    ->toArray();

            default:
                return [];
        }
    }

    /**
     * Export report data to CSV.
     */
    private function exportCsv(array $data, string $reportType): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        $filename = $reportType . '-' . now()->format('Y-m-d-His') . '.csv';

        return response()->streamDownload(function () use ($data, $reportType) {
            $handle = fopen('php://output', 'w');

            // Add headers based on report type
            switch ($reportType) {
                case 'sales_by_rep':
                    fputcsv($handle, ['Rep Name', 'Email', 'Total Orders', 'Completed Orders', 'Total Items']);
                    break;
                case 'sales_by_pharmacy':
                    fputcsv($handle, ['Pharmacy Name', 'Email', 'Total Orders', 'Total Items']);
                    break;
                case 'product_performance':
                    fputcsv($handle, ['Product Name', 'Total Quantity', 'Order Count']);
                    break;
            }

            // Add data rows
            foreach ($data as $row) {
                fputcsv($handle, $row);
            }

            fclose($handle);
        }, $filename, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * Export report data to PDF.
     */
    private function exportPdf(array $data, string $reportType, Request $request): \Illuminate\Http\Response
    {
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.report', [
            'data' => $data,
            'reportType' => $reportType,
            'dateFrom' => $request->input('date_from'),
            'dateTo' => $request->input('date_to'),
            'generatedAt' => now()->format('Y-m-d H:i:s'),
        ]);

        $filename = $reportType . '-' . now()->format('Y-m-d-His') . '.pdf';

        return $pdf->download($filename);
    }
}
