<?php

namespace App\Http\Controllers\Rep;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Pharmacy;
use App\Models\Product;
use App\Models\Wholesaler;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class OrderController extends Controller
{
    /**
     * Show the order creation form for a specific pharmacy.
     */
    public function create(Request $request, Pharmacy $pharmacy): Response
    {
        $user = $request->user();

        // Ensure pharmacy belongs to rep's company
        if ($pharmacy->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Load pharmacy with wholesaler accounts
        $pharmacy->load(['wholesalers']);

        // Get all products for this company
        $products = Product::where('company_id', $user->company_id)
            ->where('active', true)
            ->with('brand')
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('name', 'like', "%{$search}%")
                        ->orWhere('barcode', 'like', "%{$search}%")
                        ->orWhere('size', 'like', "%{$search}%");
                });
            })
            ->when($request->input('brand_id'), function ($q, $brandId) {
                $q->where('brand_id', $brandId);
            })
            ->orderBy('name')
            ->get()
            ->groupBy(function ($product) {
                return $product->brand->name ?? 'Other';
            });

        // Get all wholesalers for dropdown
        $wholesalers = Wholesaler::where('status', 'active')
            ->orderBy('wholesaler_name')
            ->get();

        return Inertia::render('Rep/Orders/Create', [
            'pharmacy' => $pharmacy,
            'products' => $products,
            'wholesalers' => $wholesalers,
            'filters' => $request->only(['search', 'brand_id']),
        ]);
    }

    /**
     * Store a newly created order.
     */
    public function store(Request $request): RedirectResponse
    {
        $user = $request->user();

        $validated = $request->validate([
            'pharmacy_id' => 'required|exists:pharmacies,id',
            'wholesaler_id' => 'required|exists:wholesalers,id',
            'wholesaler_account_number' => 'nullable|string|max:100',
            'delivery_date' => 'nullable|date',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        // Verify pharmacy belongs to rep's company
        $pharmacy = Pharmacy::findOrFail($validated['pharmacy_id']);
        if ($pharmacy->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        DB::beginTransaction();

        try {
            // Calculate total items
            $totalItems = array_sum(array_column($validated['items'], 'quantity'));

            // Create order
            $order = Order::create([
                'company_id' => $user->company_id,
                'pharmacy_id' => $validated['pharmacy_id'],
                'rep_id' => $user->id,
                'wholesaler_id' => $validated['wholesaler_id'],
                'wholesaler_account_number' => $validated['wholesaler_account_number'] ?? null,
                'delivery_date' => $validated['delivery_date'] ?? null,
                'total_items' => $totalItems,
                'status' => 'pending',
                'notes' => $validated['notes'] ?? null,
            ]);

            // Create order items with product snapshots
            foreach ($validated['items'] as $item) {
                $product = Product::with('brand')->findOrFail($item['product_id']);

                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $product->id,
                    'quantity' => $item['quantity'],
                    'product_snapshot' => [
                        'name' => $product->name,
                        'size' => $product->size,
                        'barcode' => $product->barcode,
                        'brand_name' => $product->brand->name ?? 'Unknown',
                    ],
                ]);
            }

            DB::commit();

            return redirect()
                ->route('rep.orders.show', $order)
                ->with('success', 'Order created successfully! Order #' . $order->order_number);

        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Failed to create order: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified order.
     */
    public function show(Request $request, Order $order): Response
    {
        $user = $request->user();

        // Ensure order belongs to rep's company and was created by this rep
        if ($order->company_id !== $user->company_id || $order->rep_id !== $user->id) {
            abort(403, 'Unauthorized action.');
        }

        $order->load([
            'pharmacy',
            'wholesaler',
            'items.product.brand',
        ]);

        // Group items by brand
        $itemsByBrand = $order->items->groupBy(function ($item) {
            return $item->product_snapshot['brand_name'] ?? 'Unknown';
        });

        return Inertia::render('Rep/Orders/Show', [
            'order' => $order,
            'itemsByBrand' => $itemsByBrand,
        ]);
    }

    /**
     * Display list of orders for the rep.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();

        $orders = Order::where('rep_id', $user->id)
            ->with(['pharmacy', 'wholesaler'])
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->input('pharmacy_id'), function ($q, $pharmacyId) {
                $q->where('pharmacy_id', $pharmacyId);
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Get pharmacies for filter
        $pharmacies = Pharmacy::where('company_id', $user->company_id)
            ->where('status', 'active')
            ->orderBy('pharmacy_name')
            ->get();

        return Inertia::render('Rep/Orders/Index', [
            'orders' => $orders,
            'pharmacies' => $pharmacies,
            'filters' => $request->only(['status', 'pharmacy_id']),
        ]);
    }
}
