<?php

namespace App\Http\Controllers\SupplierAdmin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Brand;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Inertia\Response;

class ProductController extends Controller
{
    /**
     * Display a listing of products for the supplier's company.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $companyId = $user->company_id;

        $products = Product::where('company_id', $companyId)
            ->with(['brand', 'company'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('name', 'like', "%{$search}%")
                        ->orWhere('barcode', 'like', "%{$search}%")
                        ->orWhere('size', 'like', "%{$search}%");
                });
            })
            ->when($request->input('brand_id'), function ($q, $brandId) {
                $q->where('brand_id', $brandId);
            })
            ->when($request->input('active') !== null, function ($q) use ($request) {
                $q->where('active', $request->input('active'));
            })
            ->orderBy('sort_order')
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Get brands for filter dropdown
        $brands = Brand::where('company_id', $companyId)
            ->orderBy('name')
            ->get();

        return Inertia::render('SupplierAdmin/Products/Index', [
            'products' => $products,
            'brands' => $brands,
            'filters' => $request->only(['search', 'brand_id', 'active']),
            'breadcrumbs' => [
                ['label' => 'Supplier Dashboard', 'url' => route('supplier-admin.dashboard')],
                ['label' => 'Products', 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for creating a new product.
     */
    public function create(Request $request): Response
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        // Get brands for the supplier's company
        $brands = Brand::where('company_id', $user->company_id)
            ->orderBy('name')
            ->get();

        return Inertia::render('SupplierAdmin/Products/Create', [
            'brands' => $brands,
            'breadcrumbs' => [
                ['label' => 'Supplier Dashboard', 'url' => route('supplier-admin.dashboard')],
                ['label' => 'Products', 'url' => route('supplier-admin.products.index')],
                ['label' => 'Create', 'url' => null],
            ],
        ]);
    }

    /**
     * Store a newly created product in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'size' => 'nullable|string|max:100',
            'price' => 'nullable|numeric|min:0|max:99999999.99',
            'barcode' => 'required|string|max:100|unique:products',
            'active' => 'required|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        // Auto-assign company_id from authenticated user
        $validated['company_id'] = $user->company_id;

        $product = Product::create($validated);

        return redirect()
            ->route('supplier-admin.products.index')
            ->with('success', 'Product created successfully.');
    }

    /**
     * Show the form for editing the specified product.
     */
    public function edit(Request $request, Product $product): Response
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        // Ensure product belongs to current company
        if ($product->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Get brands for the supplier's company
        $brands = Brand::where('company_id', $user->company_id)
            ->orderBy('name')
            ->get();

        return Inertia::render('SupplierAdmin/Products/Edit', [
            'product' => $product,
            'brands' => $brands,
            'breadcrumbs' => [
                ['label' => 'Supplier Dashboard', 'url' => route('supplier-admin.dashboard')],
                ['label' => 'Products', 'url' => route('supplier-admin.products.index')],
                ['label' => $product->name, 'url' => null],
                ['label' => 'Edit', 'url' => null],
            ],
        ]);
    }

    /**
     * Update the specified product in storage.
     */
    public function update(Request $request, Product $product): RedirectResponse
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        // Ensure product belongs to current company
        if ($product->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'size' => 'nullable|string|max:100',
            'price' => 'nullable|numeric|min:0|max:99999999.99',
            'barcode' => [
                'required',
                'string',
                'max:100',
                Rule::unique('products')->ignore($product->id),
            ],
            'active' => 'required|boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        $product->update($validated);

        return redirect()
            ->route('supplier-admin.products.index')
            ->with('success', 'Product updated successfully.');
    }

    /**
     * Soft delete the specified product from storage.
     */
    public function destroy(Request $request, Product $product): RedirectResponse
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        // Ensure product belongs to current company
        if ($product->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $product->delete();

        return redirect()
            ->route('supplier-admin.products.index')
            ->with('success', 'Product deleted successfully.');
    }

    /**
     * Show the CSV upload form.
     */
    public function upload(Request $request): Response
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        return Inertia::render('SupplierAdmin/Products/Upload', [
            'breadcrumbs' => [
                ['label' => 'Supplier Dashboard', 'url' => route('supplier-admin.dashboard')],
                ['label' => 'Products', 'url' => route('supplier-admin.products.index')],
                ['label' => 'Upload CSV', 'url' => null],
            ],
        ]);
    }

    /**
     * Process CSV file import.
     * CSV format: Brand, Product Name, Size, Barcode, Price
     */
    public function import(Request $request): RedirectResponse
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $request->validate([
            'csv_file' => 'required|file|mimes:csv,txt|max:10240', // 10MB max
        ]);

        $file = $request->file('csv_file');
        $path = $file->store('imports', 'local');

        try {
            $fullPath = Storage::path($path);
            $handle = fopen($fullPath, 'r');

            $imported = 0;
            $updated = 0;
            $skipped = 0;
            $errors = [];

            // Read header row
            $header = fgetcsv($handle);

            // Validate CSV format (should have: Brand, Product Name, Size, Barcode, Price)
            if (count($header) < 5) {
                fclose($handle);
                Storage::delete($path);
                return redirect()
                    ->back()
                    ->with('error', 'Invalid CSV format. Expected columns: Brand, Product Name, Size, Barcode, Price');
            }

            $rowNumber = 1;

            while (($row = fgetcsv($handle)) !== false) {
                $rowNumber++;

                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }

                // Extract data from CSV
                $brandName = trim($row[0] ?? '');
                $productName = trim($row[1] ?? '');
                $size = trim($row[2] ?? '');
                $barcode = trim($row[3] ?? '');
                $price = trim($row[4] ?? '');

                // Validate required fields
                if (empty($brandName) || empty($productName) || empty($barcode)) {
                    $errors[] = "Row {$rowNumber}: Missing required fields (Brand, Product Name, or Barcode)";
                    $skipped++;
                    continue;
                }

                // Validate price if provided
                if (!empty($price) && !is_numeric($price)) {
                    $errors[] = "Row {$rowNumber}: Invalid price format";
                    $skipped++;
                    continue;
                }

                try {
                    // Find or create brand
                    $brand = Brand::firstOrCreate(
                        [
                            'name' => $brandName,
                            'company_id' => $user->company_id,
                        ],
                        [
                            'active' => true,
                            'sort_order' => 0,
                        ]
                    );

                    // Create or update product
                    $product = Product::updateOrCreate(
                        [
                            'barcode' => $barcode,
                            'company_id' => $user->company_id,
                        ],
                        [
                            'brand_id' => $brand->id,
                            'name' => $productName,
                            'size' => $size ?: null,
                            'price' => !empty($price) ? $price : null,
                            'active' => true,
                            'sort_order' => 0,
                        ]
                    );

                    if ($product->wasRecentlyCreated) {
                        $imported++;
                    } else {
                        $updated++;
                    }
                } catch (\Exception $e) {
                    $errors[] = "Row {$rowNumber}: " . $e->getMessage();
                    $skipped++;
                }
            }

            fclose($handle);
            Storage::delete($path);

            $message = "Import completed: {$imported} created, {$updated} updated, {$skipped} skipped.";

            if (count($errors) > 0) {
                $message .= " Errors: " . implode('; ', array_slice($errors, 0, 5));
                if (count($errors) > 5) {
                    $message .= " (+" . (count($errors) - 5) . " more)";
                }
            }

            return redirect()
                ->route('supplier-admin.products.index')
                ->with($skipped > 0 ? 'warning' : 'success', $message);
        } catch (\Exception $e) {
            if (isset($path)) {
                Storage::delete($path);
            }

            return redirect()
                ->back()
                ->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Export products to CSV with prices.
     */
    public function export(Request $request): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        $user = $request->user();

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $companyId = $user->company_id;

        $products = Product::where('company_id', $companyId)
            ->with(['brand'])
            ->orderBy('brand_id')
            ->orderBy('sort_order')
            ->get();

        $filename = 'products-' . now()->format('Y-m-d-His') . '.csv';

        return response()->streamDownload(function () use ($products) {
            $handle = fopen('php://output', 'w');

            // Add CSV headers matching import format
            fputcsv($handle, ['Brand', 'Product Name', 'Size', 'Barcode', 'Price']);

            foreach ($products as $product) {
                fputcsv($handle, [
                    $product->brand->name ?? '',
                    $product->name,
                    $product->size,
                    $product->barcode,
                    $product->price ?? '',
                ]);
            }

            fclose($handle);
        }, $filename, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }
}
