<?php

namespace App\Http\Controllers\SupplierAdmin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Inertia\Inertia;
use Inertia\Response;

class RepController extends Controller
{
    /**
     * Display a listing of reps for the supplier's company.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Only show reps for the supplier's own company
        $query = User::where('role', 'rep')
            ->where('company_id', $companyId);

        $reps = $query
            ->with(['company'])
            ->withCount(['orders', 'pharmacies'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                });
            })
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        return Inertia::render('SupplierAdmin/Reps/Index', [
            'reps' => $reps,
            'filters' => $request->only(['search', 'status']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Sales Reps', 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for creating a new rep.
     */
    public function create(): Response
    {
        return Inertia::render('SupplierAdmin/Reps/Create', [
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Sales Reps', 'url' => route('supplier-admin.reps.index')],
                ['label' => 'Create', 'url' => null],
            ],
        ]);
    }

    /**
     * Store a newly created rep in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $user = $request->user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users',
            'phone' => 'nullable|string|max:50',
            'territory' => 'nullable|string|max:255',
            'password' => ['required', 'confirmed', Password::defaults()],
            'status' => 'required|in:active,inactive',
        ]);

        // Auto-assign company_id from authenticated supplier admin
        $validated['company_id'] = $user->company_id;
        $validated['role'] = 'rep';
        $validated['password'] = Hash::make($validated['password']);

        $rep = User::create($validated);

        return redirect()
            ->route('supplier-admin.reps.show', $rep)
            ->with('success', 'Sales rep created successfully.');
    }

    /**
     * Display the specified rep with performance statistics.
     */
    public function show(Request $request, User $rep): Response
    {
        $user = $request->user();

        // Ensure rep belongs to supplier's company
        if ($rep->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $rep->load(['company']);
        $rep->loadCount(['orders', 'pharmacies']);

        // Performance statistics
        $stats = [
            'total_orders' => $rep->orders_count,
            'completed_orders' => $rep->orders()->where('status', 'completed')->count(),
            'pending_orders' => $rep->orders()->where('status', 'pending')->count(),
            'failed_orders' => $rep->orders()->where('status', 'failed')->count(),
            'total_pharmacies' => $rep->pharmacies_count,
            'active_pharmacies' => $rep->pharmacies()->where('status', 'active')->count(),
            'orders_this_month' => $rep->orders()
                ->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->count(),
            'orders_this_week' => $rep->orders()
                ->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])
                ->count(),
        ];

        // Recent orders
        $recentOrders = $rep->orders()
            ->with(['pharmacy'])
            ->latest()
            ->limit(10)
            ->get();

        // Top pharmacies by order count
        $topPharmacies = $rep->pharmacies()
            ->withCount('orders')
            ->orderBy('orders_count', 'desc')
            ->limit(5)
            ->get();

        return Inertia::render('SupplierAdmin/Reps/Show', [
            'rep' => $rep,
            'stats' => $stats,
            'recentOrders' => $recentOrders,
            'topPharmacies' => $topPharmacies,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Sales Reps', 'url' => route('supplier-admin.reps.index')],
                ['label' => $rep->name, 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for editing the specified rep.
     */
    public function edit(Request $request, User $rep): Response
    {
        $user = $request->user();

        // Ensure rep belongs to supplier's company
        if ($rep->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        return Inertia::render('SupplierAdmin/Reps/Edit', [
            'rep' => $rep,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Sales Reps', 'url' => route('supplier-admin.reps.index')],
                ['label' => $rep->name, 'url' => route('supplier-admin.reps.show', $rep)],
                ['label' => 'Edit', 'url' => null],
            ],
        ]);
    }

    /**
     * Update the specified rep in storage.
     */
    public function update(Request $request, User $rep): RedirectResponse
    {
        $user = $request->user();

        // Ensure rep belongs to supplier's company
        if ($rep->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $rep->id,
            'phone' => 'nullable|string|max:50',
            'territory' => 'nullable|string|max:255',
            'status' => 'required|in:active,inactive',
        ]);

        $rep->update($validated);

        return redirect()
            ->route('supplier-admin.reps.show', $rep)
            ->with('success', 'Sales rep updated successfully.');
    }

    /**
     * Deactivate the specified rep (set status to inactive).
     */
    public function destroy(Request $request, User $rep): RedirectResponse
    {
        $user = $request->user();

        // Ensure rep belongs to supplier's company
        if ($rep->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        // Deactivate instead of delete
        $rep->update(['status' => 'inactive']);

        return redirect()
            ->route('supplier-admin.reps.index')
            ->with('success', 'Sales rep deactivated successfully.');
    }
}
