<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * EnsureCompanyAccess Middleware
 *
 * CRITICAL SECURITY: Verifies that model being accessed belongs to authenticated user's company.
 * This middleware provides an additional layer of multi-tenant data isolation beyond policies.
 *
 * Usage:
 * - Apply to routes that access company-scoped models (Pharmacy, Product, Order)
 * - Checks if the model's company_id matches the authenticated user's company_id
 * - Super admins bypass this check and can access all data
 *
 * Example in routes/api.php:
 * Route::middleware(['auth:sanctum', 'ensure.company.access'])->group(function () {
 *     Route::apiResource('pharmacies', PharmacyController::class);
 * });
 */
class EnsureCompanyAccess
{
    /**
     * Models that require company access verification.
     *
     * @var array
     */
    protected $scopedModels = [
        'pharmacy' => \App\Models\Pharmacy::class,
        'product' => \App\Models\Product::class,
        'order' => \App\Models\Order::class,
        'company' => \App\Models\Company::class,
    ];

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Skip check if user is not authenticated (will be handled by auth middleware)
        if (!auth()->check()) {
            return $next($request);
        }

        $user = auth()->user();

        // Super admin bypasses company access checks
        if ($user->role === 'super_admin') {
            return $next($request);
        }

        // Check route parameters for scoped models
        foreach ($this->scopedModels as $paramName => $modelClass) {
            if ($request->route($paramName)) {
                $model = $request->route($paramName);

                // Verify model has company_id property
                if (!property_exists($model, 'company_id')) {
                    continue;
                }

                // Special handling for Company model
                if ($model instanceof \App\Models\Company) {
                    // Users can only access their own company
                    if ($user->company_id !== $model->id) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Unauthorized. You do not have access to this company.',
                        ], Response::HTTP_FORBIDDEN);
                    }
                    continue;
                }

                // CRITICAL: Verify company_id matches for all other models
                if ($model->company_id !== $user->company_id) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Unauthorized. This resource does not belong to your company.',
                    ], Response::HTTP_FORBIDDEN);
                }
            }
        }

        return $next($request);
    }
}
