<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;

class Order extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'company_id',
        'rep_id',
        'pharmacy_id',
        'wholesaler_id',
        'wholesaler_account_number',
        'order_number',
        'delivery_date',
        'total_amount',
        'total_items',
        'status',
        'notes',
        'processed_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'company_id' => 'integer',
        'rep_id' => 'integer',
        'pharmacy_id' => 'integer',
        'wholesaler_id' => 'integer',
        'total_amount' => 'decimal:2',
        'total_items' => 'integer',
        'delivery_date' => 'date',
        'processed_at' => 'datetime',
    ];

    /**
     * The "booted" method of the model.
     */
    protected static function booted(): void
    {
        // Global scope for company isolation - DISABLED until migrations are run

        // Auto-generate order number on creating
        static::creating(function ($order) {
            if (empty($order->order_number) && \Schema::hasColumn('orders', 'order_number')) {
                $order->order_number = self::generateOrderNumber();
            }
        });
    }

    /**
     * Generate a unique order number.
     */
    protected static function generateOrderNumber(): string
    {
        do {
            $orderNumber = 'ORD-' . strtoupper(Str::random(8));
        } while (self::where('order_number', $orderNumber)->exists());

        return $orderNumber;
    }

    /**
     * Get the company that owns the order.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the rep that created the order.
     */
    public function rep(): BelongsTo
    {
        return $this->belongsTo(User::class, 'rep_id');
    }

    /**
     * Get the pharmacy that owns the order.
     */
    public function pharmacy(): BelongsTo
    {
        return $this->belongsTo(Pharmacy::class);
    }

    /**
     * Get the wholesaler for the order.
     */
    public function wholesaler(): BelongsTo
    {
        return $this->belongsTo(Wholesaler::class);
    }

    /**
     * Get the items for the order.
     */
    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get the PDF for the order.
     */
    public function pdf(): HasOne
    {
        return $this->hasOne(OrderPdf::class);
    }

    /**
     * Scope a query to filter by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to filter by pharmacy.
     */
    public function scopeByPharmacy($query, int $pharmacyId)
    {
        return $query->where('pharmacy_id', $pharmacyId);
    }

    /**
     * Scope a query to filter by rep.
     */
    public function scopeByRep($query, int $repId)
    {
        return $query->where('rep_id', $repId);
    }

    /**
     * Scope a query to filter by date range.
     */
    public function scopeDateRange($query, $startDate, $endDate = null)
    {
        $query->whereDate('created_at', '>=', $startDate);

        if ($endDate) {
            $query->whereDate('created_at', '<=', $endDate);
        }

        return $query;
    }

    /**
     * Mark order as processing.
     */
    public function markAsProcessing(): void
    {
        $this->update(['status' => 'processing']);
    }

    /**
     * Mark order as completed.
     */
    public function markAsCompleted(): void
    {
        $this->update([
            'status' => 'completed',
            'processed_at' => now(),
        ]);
    }

    /**
     * Mark order as failed.
     */
    public function markAsFailed(string $notes = null): void
    {
        $this->update([
            'status' => 'failed',
            'notes' => $notes,
        ]);
    }

    /**
     * Calculate total items from order items.
     */
    public function calculateTotalItems(): int
    {
        return $this->items()->sum('quantity');
    }

    /**
     * Update total items count.
     */
    public function updateTotalItems(): void
    {
        $this->update(['total_items' => $this->calculateTotalItems()]);
    }
}
