<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Order;

/**
 * OrderPolicy
 *
 * Authorization policy for Order model.
 * Enforces multi-tenant data isolation - users can ONLY access orders from their company.
 *
 * Authorization Rules:
 * - super_admin: Can perform all actions on any order
 * - company_admin: Can view, update, and delete all orders in their company
 * - rep: Can view and create orders, can update only their own orders
 *
 * CRITICAL: All operations must verify company_id matches to prevent cross-company access.
 */
class OrderPolicy
{
    /**
     * Determine if the user can view any orders.
     *
     * All authenticated users can view orders from their company.
     * Orders should be filtered by company_id in queries.
     *
     * @param User $user
     * @return bool
     */
    public function viewAny(User $user): bool
    {
        // All authenticated users can view orders (scoped to their company)
        return true;
    }

    /**
     * Determine if the user can view a specific order.
     *
     * Users can only view orders that belong to their company.
     * CRITICAL: This prevents cross-company data access.
     *
     * @param User $user
     * @param Order $order
     * @return bool
     */
    public function view(User $user, Order $order): bool
    {
        // Super admin can view any order
        if ($user->role === 'super_admin') {
            return true;
        }

        // CRITICAL: User can only view if order belongs to their company
        return $user->company_id === $order->company_id;
    }

    /**
     * Determine if the user can create an order.
     *
     * All authenticated users from the same company can create orders.
     * The order will be automatically scoped to their company_id during creation.
     *
     * @param User $user
     * @return bool
     */
    public function create(User $user): bool
    {
        // All authenticated users can create orders (auto-scoped to their company)
        return true;
    }

    /**
     * Determine if the user can update an order.
     *
     * Company admins can update any order from their company.
     * Reps can only update orders they created.
     * Super admins can update any order.
     *
     * @param User $user
     * @param Order $order
     * @return bool
     */
    public function update(User $user, Order $order): bool
    {
        // Super admin can update any order
        if ($user->role === 'super_admin') {
            return true;
        }

        // CRITICAL: Order must belong to user's company
        if ($user->company_id !== $order->company_id) {
            return false;
        }

        // Company admin can update any order from their company
        if ($user->role === 'company_admin') {
            return true;
        }

        // Rep can only update orders they created
        if ($user->role === 'rep') {
            return $order->rep_id === $user->id;
        }

        return false;
    }

    /**
     * Determine if the user can delete an order.
     *
     * Only company admins and super admins can delete orders.
     * Reps cannot delete orders.
     *
     * @param User $user
     * @param Order $order
     * @return bool
     */
    public function delete(User $user, Order $order): bool
    {
        // Super admin can delete any order
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can delete orders from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $order->company_id;
        }

        // Reps cannot delete orders
        return false;
    }

    /**
     * Determine if the user can restore a soft-deleted order.
     *
     * Only company admins and super admins can restore orders.
     *
     * @param User $user
     * @param Order $order
     * @return bool
     */
    public function restore(User $user, Order $order): bool
    {
        // Super admin can restore any order
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can restore orders from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $order->company_id;
        }

        return false;
    }

    /**
     * Determine if the user can permanently delete an order.
     *
     * Only super admins can force delete orders.
     *
     * @param User $user
     * @param Order $order
     * @return bool
     */
    public function forceDelete(User $user, Order $order): bool
    {
        return $user->role === 'super_admin';
    }

    /**
     * Determine if the user can resend order email.
     *
     * Company admins and order creators can resend emails.
     *
     * @param User $user
     * @param Order $order
     * @return bool
     */
    public function resendEmail(User $user, Order $order): bool
    {
        // Super admin can resend any order email
        if ($user->role === 'super_admin') {
            return true;
        }

        // CRITICAL: Order must belong to user's company
        if ($user->company_id !== $order->company_id) {
            return false;
        }

        // Company admin can resend emails for their company's orders
        if ($user->role === 'company_admin') {
            return true;
        }

        // Rep can resend emails for orders they created
        if ($user->role === 'rep') {
            return $order->rep_id === $user->id;
        }

        return false;
    }

    /**
     * Determine if the user can regenerate order PDF.
     *
     * Company admins and order creators can regenerate PDFs.
     *
     * @param User $user
     * @param Order $order
     * @return bool
     */
    public function regeneratePDF(User $user, Order $order): bool
    {
        // Super admin can regenerate any order PDF
        if ($user->role === 'super_admin') {
            return true;
        }

        // CRITICAL: Order must belong to user's company
        if ($user->company_id !== $order->company_id) {
            return false;
        }

        // Company admin can regenerate PDFs for their company's orders
        if ($user->role === 'company_admin') {
            return true;
        }

        // Rep can regenerate PDFs for orders they created
        if ($user->role === 'rep') {
            return $order->rep_id === $user->id;
        }

        return false;
    }
}
