<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Product;

/**
 * ProductPolicy
 *
 * Authorization policy for Product model.
 * Enforces multi-tenant data isolation - users can ONLY access products from their company.
 *
 * Authorization Rules:
 * - super_admin: Can perform all actions on any product
 * - company_admin: Can manage all products in their company (create, update, delete)
 * - rep: Can only view products from their company, cannot modify
 *
 * CRITICAL: All operations must verify company_id matches to prevent cross-company access.
 */
class ProductPolicy
{
    /**
     * Determine if the user can view any products.
     *
     * All authenticated users can view products from their company.
     * Products should be filtered by company_id in queries.
     *
     * @param User $user
     * @return bool
     */
    public function viewAny(User $user): bool
    {
        // All authenticated users can view products (scoped to their company)
        return true;
    }

    /**
     * Determine if the user can view a specific product.
     *
     * Users can only view products that belong to their company.
     * CRITICAL: This prevents cross-company data access.
     *
     * @param User $user
     * @param Product $product
     * @return bool
     */
    public function view(User $user, Product $product): bool
    {
        // Super admin can view any product
        if ($user->role === 'super_admin') {
            return true;
        }

        // CRITICAL: User can only view if product belongs to their company
        return $user->company_id === $product->company_id;
    }

    /**
     * Determine if the user can create a product.
     *
     * Only company admins and super admins can create products.
     * Reps cannot create products - they can only use existing catalog.
     *
     * @param User $user
     * @return bool
     */
    public function create(User $user): bool
    {
        // Super admin can create products for any company
        if ($user->role === 'super_admin') {
            return true;
        }

        // Only company admins can create products
        return $user->role === 'company_admin';
    }

    /**
     * Determine if the user can update a product.
     *
     * Only company admins can update products from their company.
     * Reps cannot modify product catalog.
     *
     * @param User $user
     * @param Product $product
     * @return bool
     */
    public function update(User $user, Product $product): bool
    {
        // Super admin can update any product
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can only update products from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $product->company_id;
        }

        // Reps cannot update products
        return false;
    }

    /**
     * Determine if the user can delete a product.
     *
     * Only company admins can delete products from their company.
     * Reps cannot delete products.
     *
     * @param User $user
     * @param Product $product
     * @return bool
     */
    public function delete(User $user, Product $product): bool
    {
        // Super admin can delete any product
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can only delete products from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $product->company_id;
        }

        // Reps cannot delete products
        return false;
    }

    /**
     * Determine if the user can restore a soft-deleted product.
     *
     * Only company admins and super admins can restore products.
     *
     * @param User $user
     * @param Product $product
     * @return bool
     */
    public function restore(User $user, Product $product): bool
    {
        // Super admin can restore any product
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can restore products from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $product->company_id;
        }

        return false;
    }

    /**
     * Determine if the user can permanently delete a product.
     *
     * Only super admins can force delete products.
     *
     * @param User $user
     * @param Product $product
     * @return bool
     */
    public function forceDelete(User $user, Product $product): bool
    {
        return $user->role === 'super_admin';
    }

    /**
     * Determine if the user can activate/deactivate a product.
     *
     * Only company admins can change product active status.
     *
     * @param User $user
     * @param Product $product
     * @return bool
     */
    public function toggleActive(User $user, Product $product): bool
    {
        // Super admin can toggle any product
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can toggle products from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $product->company_id;
        }

        return false;
    }
}
