<?php

use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\BrandController;
use App\Http\Controllers\Api\ProductController;
use App\Http\Controllers\Api\OrderController;
use App\Http\Controllers\Api\PharmacyController;
use App\Http\Controllers\Api\WholesalerController;
use App\Http\Controllers\Api\PharmacyWholesalerController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// API Version 1
Route::prefix('v1')->group(function () {

    // Public routes (no authentication required)

    // Authentication routes
    Route::post('/auth/login', [AuthController::class, 'login']);

    // Legacy endpoint for parsing products (compatibility with old frontend)
    Route::get('/parse-products', function () {
        $brands = \App\Models\Brand::active()
            ->ordered()
            ->with(['activeProducts' => function ($q) {
                $q->orderBy('sort_order')->orderBy('name');
            }])
            ->get();

        // Format response to match old CSV parser format
        $result = [];
        foreach ($brands as $brand) {
            $result[$brand->name] = $brand->activeProducts->map(function ($product) {
                return [
                    'name' => $product->name,
                    'size' => $product->size,
                    'barcode' => $product->barcode,
                ];
            })->toArray();
        }

        return response()->json($result);
    });

    // Protected routes (require authentication)
    Route::middleware('auth:sanctum')->group(function () {

        // Auth routes
        Route::post('/auth/logout', [AuthController::class, 'logout']);
        Route::get('/auth/me', [AuthController::class, 'me']);

        // Brands routes
        Route::prefix('brands')->group(function () {
            Route::get('/', [BrandController::class, 'index']);
            Route::get('/{brand}', [BrandController::class, 'show']);
            Route::post('/', [BrandController::class, 'store']);
            Route::put('/{brand}', [BrandController::class, 'update']);
            Route::delete('/{brand}', [BrandController::class, 'destroy']);
        });

        // Products routes
        Route::prefix('products')->group(function () {
            Route::get('/', [ProductController::class, 'index']);
            Route::get('/{product}', [ProductController::class, 'show']);
            Route::post('/', [ProductController::class, 'store']);
            Route::put('/{product}', [ProductController::class, 'update']);
            Route::delete('/{product}', [ProductController::class, 'destroy']);

            // Legacy compatibility
            Route::get('/brand/{brandName}', [ProductController::class, 'byBrandName']);
        });

        // Pharmacies routes
        Route::prefix('pharmacies')->group(function () {
            Route::get('/', [PharmacyController::class, 'index']);
            Route::post('/', [PharmacyController::class, 'store']);
            Route::get('/{pharmacy}', [PharmacyController::class, 'show']);
            Route::put('/{pharmacy}', [PharmacyController::class, 'update']);
            Route::delete('/{pharmacy}', [PharmacyController::class, 'destroy']);
        });

        // Wholesalers routes
        Route::prefix('wholesalers')->group(function () {
            Route::get('/', [WholesalerController::class, 'index']);
            Route::get('/{wholesaler}', [WholesalerController::class, 'show']);
        });

        // Pharmacy-Wholesaler accounts routes
        Route::prefix('pharmacy-wholesaler-accounts')->group(function () {
            Route::post('/', [PharmacyWholesalerController::class, 'store']);
            Route::put('/{pharmacyWholesaler}', [PharmacyWholesalerController::class, 'update']);
            Route::delete('/{pharmacyWholesaler}', [PharmacyWholesalerController::class, 'destroy']);
        });

        // Orders routes
        Route::prefix('orders')->group(function () {
            Route::get('/', [OrderController::class, 'index']);
            Route::post('/', [OrderController::class, 'store']);
            Route::get('/{order}', [OrderController::class, 'show']);
            Route::put('/{order}/status', [OrderController::class, 'updateStatus']);
            Route::post('/{order}/resend-email', [OrderController::class, 'resendEmail']);
            Route::post('/{order}/regenerate-pdf', [OrderController::class, 'regeneratePDF']);
        });
    });
});

// Health check endpoint
Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'timestamp' => now()->toISOString(),
        'version' => '1.0.0',
    ]);
});
