<?php

namespace App\Services;

use App\Models\Order;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class EmailService
{
    /**
     * Send order confirmation email.
     */
    public function sendOrderEmail(Order $order): bool
    {
        try {
            $order->load(['items.product.brand', 'pdf']);

            // Prepare email data
            $emailData = [
                'order' => $order,
                'pharmacy' => $order->pharmacy_name ?? 'N/A',
                'totalItems' => $order->total_items,
                'items' => $order->items,
                'itemsByBrand' => $order->items->groupBy(function ($item) {
                    return $item->getProductDetails()['brand_name'];
                }),
            ];

            // Send email
            Mail::send('emails.order-confirmation', $emailData, function ($message) use ($order) {
                $message->to($order->email)
                    ->subject($this->getEmailSubject($order))
                    ->from(config('mail.from.address'), config('mail.from.name'));

                // Attach PDF if it exists
                if ($order->pdf && $order->pdf->exists()) {
                    $message->attach($order->pdf->full_path, [
                        'as' => $order->pdf->file_name,
                        'mime' => 'application/pdf',
                    ]);
                }
            });

            Log::info("Order email sent successfully", [
                'order_id' => $order->id,
                'email' => $order->email,
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error("Failed to send order email", [
                'order_id' => $order->id,
                'email' => $order->email,
                'error' => $e->getMessage(),
            ]);

            return false;
        }
    }

    /**
     * Get email subject line.
     */
    protected function getEmailSubject(Order $order): string
    {
        if ($order->pharmacy_name) {
            return "Haleon Order - {$order->pharmacy_name}";
        }

        return "Haleon Product Order #{$order->id}";
    }

    /**
     * Send order failure notification.
     */
    public function sendOrderFailureNotification(Order $order, string $reason): bool
    {
        try {
            Mail::send('emails.order-failed', [
                'order' => $order,
                'reason' => $reason,
            ], function ($message) use ($order) {
                $message->to($order->email)
                    ->subject("Order Processing Failed - Order #{$order->id}")
                    ->from(config('mail.from.address'), config('mail.from.name'));
            });

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to send failure notification", [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);

            return false;
        }
    }
}
